// --- スタイルとHTMLの注入 ---

(function() {
    // 既存の要素があれば削除して終了（トグルOFF機能）
    const existingContainer = document.getElementById('time-bar-container');
    if (existingContainer) {
        existingContainer.remove();
        const existingOverlay = document.getElementById('overlay');
        if (existingOverlay) existingOverlay.remove();
        const existingModal = document.getElementById('settings-modal');
        if (existingModal) existingModal.remove();
        const existingSiteFix = document.getElementById('global-time-scale-site-fixes');
        if (existingSiteFix) existingSiteFix.remove();

        // ページの余白をリセット
        document.documentElement.style.marginTop = '';
        document.documentElement.style.marginBottom = '';
        return;
    }

// CSSの注入
const style = document.createElement('style');
style.textContent = `
    :root {
        --bg-color: rgba(30, 30, 35, 0.95);
        --text-color: #e0e0e0;
        --accent-color: #4a90e2;
        --pin-color: #ff6b6b;
        --height: 30px;
    }
    #time-bar-container {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: var(--height);
        background-color: var(--bg-color);
        color: var(--text-color);
        box-shadow: 0 4px 10px rgba(0, 0, 0, 0.3);
        z-index: 2147483647; /* 最前面 */
        overflow: hidden;
        user-select: none;
        cursor: grab;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        font-size: 12px;
        box-sizing: border-box;
    }
    #time-bar-container * { box-sizing: border-box; }
    
    #scale-layer, #pins-layer {
        position: absolute;
        top: 0; left: 0; width: 100%; height: 100%;
        pointer-events: none;
    }
    .tick {
        position: absolute; bottom: 0; width: 1px; height: 100%;
        background-color: transparent;
    }
    .tick.midnight { background-color: rgba(255, 255, 255, 0.1); }
    .tick-label {
        position: absolute; bottom: 2px; left: 4px;
        font-size: 12px; color: #aaa; white-space: nowrap;
    }
    .date-label {
        position: absolute; top: 2px; left: 6px;
        font-size: 10px; font-weight: bold; color: var(--accent-color); white-space: nowrap;
    }
    .pin {
        position: absolute; top: 0; bottom: 0; width: 2px;
        background: linear-gradient(to bottom, var(--pin-color) 0%, transparent 80%);
        display: flex; justify-content: center; align-items: center;
    }
    .pin-badge {
        background-color: var(--pin-color); color: white;
        padding: 3px 8px; border-radius: 4px;
        font-size: 11px; font-weight: bold; white-space: nowrap;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.5);
        display: flex; gap: 6px;
    }
    
    /* 最小化時のスタイル */
    #time-bar-container.collapsed {
        height: 5px !important;
        cursor: pointer;
        transition: background-color 0.2s;
    }
    #time-bar-container.collapsed > * { display: none; }
    #time-bar-container.collapsed:hover { background-color: var(--accent-color); }

    /* ボタン類 */
    #ui-controls {
        position: absolute; top: 5px; right: 10px;
        display: flex; align-items: center; gap: 5px;
        z-index: 10000;
    }
    #time-bar-container .control-group {
        display: none; gap: 5px;
    }
    #time-bar-container #ui-controls:hover .control-group { display: flex; }
    
    .icon-btn {
        background: rgba(255, 255, 255, 0.1); border: 1px solid #666;
        color: #fff; 
        width: 24px; height: 24px; padding: 0; margin: 0;
        display: flex; justify-content: center; align-items: center;
        cursor: pointer; border-radius: 4px; pointer-events: auto; font-size: 14px;
        line-height: 1; overflow: hidden; text-indent: 0;
    }
    .icon-btn:hover { background: rgba(255, 255, 255, 0.3); }

    /* 下部固定時のスタイル */
    #time-bar-container.bottom-docked {
        top: auto;
        bottom: 0;
        box-shadow: 0 -4px 10px rgba(0, 0, 0, 0.3);
    }

    /* モーダル */
    #overlay {
        display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%;
        background: rgba(0, 0, 0, 0.5); z-index: 2147483646;
    }
    #settings-modal {
        display: none; position: fixed; top: 50%; left: 50%;
        transform: translate(-50%, -50%);
        background: #2d2d33; color: #fff; padding: 20px;
        border-radius: 8px; box-shadow: 0 10px 25px rgba(0, 0, 0, 0.5);
        z-index: 2147483647; width: 300px; max-height: 80vh; overflow-y: auto;
        font-family: sans-serif;
    }
    .zone-item {
        display: flex; justify-content: space-between; align-items: center;
        padding: 8px 0; border-bottom: 1px solid #444;
    }
    #settings-modal input, #settings-modal select, #settings-modal button {
        padding: 5px; border-radius: 3px; border: 1px solid #555;
        background: #444; color: white; margin: 0;
    }
    #settings-modal button { cursor: pointer; background: #555; }
    #settings-modal button:hover { background: #666; }
    #settings-modal button.primary { background: var(--accent-color); border-color: var(--accent-color); }
    #settings-modal button.danger { background: #d9534f; border-color: #d93432; }

    /* Calendar View Styles */
    .date-label.interactive { cursor: pointer; pointer-events: auto; z-index: 100; }
    .date-label.interactive:hover { text-decoration: underline; color: #fff; }
    #main-date-label { position: absolute; top: 2px; left: 10px; }
    .calendar-day {
        position: absolute; bottom: 2px;
        text-align: center; font-size: 12px;
        display: flex; flex-direction: column; justify-content: flex-end;
    }
    .calendar-day .day-name { font-size: 9px; opacity: 0.7; text-transform: uppercase; }
    .calendar-day .day-num { font-weight: bold; font-size: 14px; line-height: 1.1; }
    .month-separator {
        position: absolute; top: 5px; bottom: 5px; width: 1px; background: rgba(255,255,255,0.2);
    }
    .month-label-large {
        position: absolute; top: 0; bottom: 0;
        font-size: 20px; font-weight: 900; color: var(--accent-color);
        display: flex; align-items: center; letter-spacing: 1px;
    }
    .calendar-day.today {
        background: rgba(255, 255, 255, 0.15);
        border: 1px solid rgba(255, 255, 255, 0.3);
        border-radius: 4px;
    }
`;
document.head.appendChild(style);

const siteFixStyle = document.createElement('style');
siteFixStyle.id = 'global-time-scale-site-fixes';
document.head.appendChild(siteFixStyle);

// HTML構造の注入
const container = document.createElement('div');
container.id = 'time-bar-container';
container.innerHTML = `
    <div id="scale-layer"></div>
    <div id="pins-layer"></div>
    <div id="main-date-label" class="date-label interactive"></div>
    <div id="ui-controls">
        <div class="control-group">
            <button id="calendar-btn" class="icon-btn" title="Calendar View">📅</button>
            <button id="position-btn" class="icon-btn" title="Move to Bottom">▼</button>
            <button id="minimize-btn" class="icon-btn" title="Minimize">_</button>
            <button id="settings-btn" class="icon-btn" title="Settings">⚙</button>
        </div>
        <button id="menu-btn" class="icon-btn">☰</button>
    </div>
`;
document.body.appendChild(container);

const overlay = document.createElement('div');
overlay.id = 'overlay';
document.body.appendChild(overlay);

const modal = document.createElement('div');
modal.id = 'settings-modal';
modal.innerHTML = `
    <h3 style="margin-top:0;">Regions</h3>
    <div id="zone-list"></div>
    <div style="margin-top: 15px; border-top: 1px solid #555; padding-top: 10px;">
        <label style="display:block; margin-bottom:5px;">Add Region:</label>
        <input type="text" id="new-label" placeholder="Label" style="width: 100px;">
        <input type="color" id="new-color" value="#ff6b6b" style="width: 30px; height: 30px; padding: 0; border: none; vertical-align: middle;">
        <select id="new-timezone" style="width: 140px;"></select>
        <button id="add-zone-btn" class="primary" style="margin-top: 5px; width: 100%;">Add</button>
    </div>
    <div style="margin-top: 15px; border-top: 1px solid #555; padding-top: 10px;">
        <label style="display:block; margin-bottom:5px;">Background Color:</label>
        <input type="color" id="bg-color-picker" value="#1e1e23" style="width: 100%; height: 30px; padding: 0; border: none; cursor: pointer;">
    </div>
    <div style="margin-top:15px; display:flex; gap:10px;">
        <button id="close-settings-btn">Close</button>
    </div>
`;
document.body.appendChild(modal);

// --- ロジック (index.htmlから移植) ---

let PIXELS_PER_HOUR = 80;

let activeZones = [
    { label: 'California', zone: 'America/Los_Angeles', color: '#ff6b6b' },
    { label: 'New York', zone: 'America/New_York', color: '#4a90e2' },
    { label: 'London', zone: 'Europe/London', color: '#50e3c2' },
    { label: '上海', zone: 'Asia/Shanghai', color: '#FF0000' },
    { label: '東京', zone: 'Asia/Tokyo', color: '#f5a623' },
];

const availableTimezones = [
    "Pacific/Honolulu", "America/Anchorage", "America/Los_Angeles", "America/Denver",
    "America/Chicago", "America/New_York", "America/Sao_Paulo", "UTC",
    "Europe/London", "Europe/Paris", "Europe/Moscow", "Asia/Dubai",
    "Asia/Kolkata", "Asia/Bangkok", "Asia/Shanghai", "Asia/Tokyo",
    "Australia/Sydney", "Pacific/Auckland"
];

const scaleLayer = document.getElementById('scale-layer');
const pinsLayer = document.getElementById('pins-layer');
const zoneListEl = document.getElementById('zone-list');
const mainDateLabel = document.getElementById('main-date-label');
const tzSelect = document.getElementById('new-timezone');
const bgColorPicker = document.getElementById('bg-color-picker');

const STORAGE_KEY_ZONES = 'globalTimeScale_activeZones';
const STORAGE_KEY_BG = 'globalTimeScale_bgColor';
const STORAGE_KEY_COLLAPSED = 'globalTimeScale_isCollapsed';
const STORAGE_KEY_POSITION = 'globalTimeScale_isBottom';

let manualOffsetMs = 0;
let isDragging = false;
let isCalendarView = false;
let calendarOffsetPx = 0;
let dragStartCalendarOffsetPx = 0;
let dragStartX = 0;
let dragStartOffsetMs = 0;
let revertTimer = null;

function storageGet(key, defaultValue) {
    return new Promise((resolve) => {
        if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
            try {
                chrome.storage.local.get([key], (result) => {
                    if (chrome.runtime.lastError) {
                        resolve(defaultValue);
                        return;
                    }
                    const value = Object.prototype.hasOwnProperty.call(result, key) ? result[key] : defaultValue;
                    resolve(value);
                });
                return;
            } catch (e) {
                // Extension context invalidated
            }
        }

        const raw = localStorage.getItem(key);
        if (raw === null) {
            resolve(defaultValue);
            return;
        }
        try {
            resolve(JSON.parse(raw));
        } catch (e) {
            resolve(raw);
        }
    });
}

function storageSet(key, value) {
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
        try {
            chrome.storage.local.set({ [key]: value });
            return;
        } catch (e) {
            // Extension context invalidated
        }
    }
    localStorage.setItem(key, JSON.stringify(value));
}

async function loadSettings() {
    const savedZones = await storageGet(STORAGE_KEY_ZONES, null);
    if (Array.isArray(savedZones) && savedZones.length > 0) {
        activeZones = savedZones;
    }

    const savedBg = await storageGet(STORAGE_KEY_BG, null);
    if (savedBg) {
        container.style.setProperty('--bg-color', savedBg);
        if (bgColorPicker) bgColorPicker.value = savedBg;
    }
}

function getUTCOffsetHours(timeZone) {
    const now = new Date();
    const isoString = now.toLocaleString('en-US', { timeZone, hour12: false });
    const localDate = new Date(isoString);
    const utcDate = new Date(now.toLocaleString('en-US', { timeZone: 'UTC', hour12: false }));
    return (localDate - utcDate) / (1000 * 60 * 60);
}

function getCenterOffset() {
    if (activeZones.length === 0) return 0;
    let total = 0;
    activeZones.forEach(z => { total += getUTCOffsetHours(z.zone); });
    return total / activeZones.length;
}

function renderCalendar() {
    scaleLayer.innerHTML = '';
    pinsLayer.innerHTML = '';
    mainDateLabel.style.display = 'none'; // Hide main label in calendar view

    const width = container.clientWidth;
    const dayWidth = 45;
    
    // Calculate visible range based on offset
    // x = 10 + calendarOffsetPx + (n * dayWidth)
    // We want to render if x + dayWidth > 0 and x < width
    const startN = Math.floor((-10 - calendarOffsetPx - dayWidth) / dayWidth);
    const endN = Math.ceil((width - 10 - calendarOffsetPx) / dayWidth);

    const today = new Date();
    today.setHours(0, 0, 0, 0);

    for (let n = startN; n <= endN; n++) {
        const currentDate = new Date(today);
        currentDate.setDate(today.getDate() + n);
        
        const x = 10 + calendarOffsetPx + (n * dayWidth);
        const dayOfWeek = currentDate.getDay(); // 0=Sun, 6=Sat
        const dateNum = currentDate.getDate();
        const isWeekend = (dayOfWeek === 0 || dayOfWeek === 6);
        const color = isWeekend ? '#ff6b6b' : '#e0e0e0';

        if (dateNum === 1) {
            const sep = document.createElement('div');
            sep.className = 'month-separator';
            sep.style.left = `${x - 5}px`;
            scaleLayer.appendChild(sep);

            const mLabel = document.createElement('div');
            mLabel.className = 'month-label-large';
            mLabel.innerText = currentDate.toLocaleDateString('en-US', { month: 'short' }).toUpperCase();
            mLabel.style.left = `${x}px`;
            scaleLayer.appendChild(mLabel);
        } else if (n === startN) {
            const mLabel = document.createElement('div');
            mLabel.className = 'date-label';
            mLabel.innerText = currentDate.toLocaleDateString('en-US', { month: 'short' });
            mLabel.style.left = `${x}px`;
            scaleLayer.appendChild(mLabel);
        }

        const dayEl = document.createElement('div');
        dayEl.className = 'calendar-day';
        if (n === 0) dayEl.classList.add('today');
        dayEl.style.left = `${x}px`;
        dayEl.style.width = `${dayWidth}px`;
        dayEl.style.color = color;
        
        if (dateNum === 1) {
            dayEl.style.paddingLeft = '35px';
        }

        dayEl.innerHTML = `
            <div class="day-name">${currentDate.toLocaleDateString('en-US', { weekday: 'short' })}</div>
            <div class="day-num">${dateNum}</div>
        `;
        scaleLayer.appendChild(dayEl);
    }
}

function render() {
    if (isCalendarView) {
        renderCalendar();
        return;
    }
    mainDateLabel.style.display = 'block';

    // ★レスポンシブ対応: 画面幅に合わせてスケールを動的に更新
    if (container.clientWidth > 0) {
        PIXELS_PER_HOUR = container.clientWidth / 24;
    }

    const now = new Date(Date.now() + manualOffsetMs);
    const utcNow = now.getTime();
    const width = container.clientWidth;
    const centerOffset = getCenterOffset();
    const screenCenterX = width / 2;

    // ピンの描画
    pinsLayer.innerHTML = '';
    activeZones.forEach(z => {
        const offset = getUTCOffsetHours(z.zone);
        const x = screenCenterX + (offset - centerOffset) * PIXELS_PER_HOUR;
        if (x < -50 || x > width + 50) return;

        const pinEl = document.createElement('div');
        pinEl.className = 'pin';
        pinEl.style.left = `${x}px`;
        pinEl.style.setProperty('--pin-color', z.color || '#ff6b6b');

        const formatter = new Intl.DateTimeFormat('en-US', {
            timeZone: z.zone, hour: '2-digit', minute: '2-digit', hour12: false
        });
        const timeString = formatter.format(now);

        pinEl.innerHTML = `
            <div class="pin-badge">
                <span>${z.label}</span>
                <span>${timeString}</span>
            </div>
        `;
        pinsLayer.appendChild(pinEl);
    });

    // スケールの描画
    scaleLayer.innerHTML = '';
    const offsetLeft = (0 - screenCenterX) / PIXELS_PER_HOUR + centerOffset;
    const offsetRight = (width - screenCenterX) / PIXELS_PER_HOUR + centerOffset;
    const currentUtcHours = now.getUTCHours() + now.getUTCMinutes() / 60 + now.getUTCSeconds() / 3600;

    const tLeft = currentUtcHours + offsetLeft;
    let mod = tLeft % 24;
    if (mod < 0) mod += 24;
    const distToNextMidnight = 24 - mod;

    // Update the static main date label
    if (distToNextMidnight > 0.5) {
        const leftDateMs = utcNow + (offsetLeft * 3600 * 1000);
        const leftDate = new Date(leftDateMs);
        mainDateLabel.innerText = leftDate.toLocaleDateString('en-US', { month: 'short', day: 'numeric', weekday: 'short', timeZone: 'UTC' });
        mainDateLabel.style.display = 'block';
    } else {
        mainDateLabel.style.display = 'none';
    }

    const startIdx = Math.floor(currentUtcHours + offsetLeft) - 1;
    const endIdx = Math.ceil(currentUtcHours + offsetRight) + 1;

    for (let h = startIdx; h <= endIdx; h++) {
        const x = (h - currentUtcHours - centerOffset) * PIXELS_PER_HOUR + screenCenterX;
        let isMidnight = false;
        let displayHour = h % 24;
        if (displayHour < 0) displayHour += 24;
        if (displayHour === 0) isMidnight = true;

        const tick = document.createElement('div');
        tick.className = `tick ${isMidnight ? 'midnight' : 'major'}`;
        tick.style.left = `${x}px`;
        scaleLayer.appendChild(tick);

        const label = document.createElement('div');
        label.className = 'tick-label';
        label.innerText = `${displayHour}:00`;
        tick.appendChild(label);

        if (isMidnight) {
            const tickDateMs = utcNow + (h - currentUtcHours) * 3600 * 1000;
            const d = new Date(tickDateMs);
            const dateStr = d.toLocaleDateString('en-US', { month: 'short', day: 'numeric', weekday: 'short', timeZone: 'UTC' });
            const dateLabel = document.createElement('div');
            dateLabel.className = 'date-label interactive';
            dateLabel.innerText = dateStr;
            dateLabel.onmousedown = (e) => e.stopPropagation();
            dateLabel.onclick = (e) => {
                e.stopPropagation();
                isCalendarView = true;
                render();
            };
            tick.appendChild(dateLabel);
        }

        const x30 = x + (0.5 * PIXELS_PER_HOUR);
        const tick30 = document.createElement('div');
        tick30.className = 'tick';
        tick30.style.height = '10px';
        tick30.style.left = `${x30}px`;
        scaleLayer.appendChild(tick30);
    }
}

function loop() {
    render();
    requestAnimationFrame(loop);
}

// --- 最小化ロジック ---
const minimizeBtn = document.getElementById('minimize-btn');
const positionBtn = document.getElementById('position-btn');
const calendarBtn = document.getElementById('calendar-btn');

function toggleCollapse() {
    const isCollapsed = container.classList.toggle('collapsed');
    storageSet(STORAGE_KEY_COLLAPSED, isCollapsed);
    updateBodyMargin();
}

// ページのコンテンツを下にずらす処理
function updateBodyMargin() {
    const height = container.classList.contains('collapsed') ? '5px' : '30px';
    const isBottom = container.classList.contains('bottom-docked');
    
    document.documentElement.style.marginTop = '';
    document.documentElement.style.marginBottom = '';
    siteFixStyle.textContent = '';

    if (isBottom) {
        document.documentElement.style.marginBottom = height;
    } else {
        document.documentElement.style.marginTop = height;
        // YouTube, LinkedIn, Facebook, X (Twitter) などの固定ヘッダーを強制的にずらす
        siteFixStyle.textContent = `
            ytd-masthead, #masthead-container { top: ${height} !important; }
            .global-nav, #global-nav { top: ${height} !important; }
            div[role="banner"], header[role="banner"] { top: ${height} !important; }
            [data-testid="primaryColumn"] > div > div:first-child { top: ${height} !important; }
            [data-testid="sidebarColumn"] > div > div:first-child { top: ${height} !important; }
        `;
    }
}

// ★最小化状態の復元
// --- 位置切り替えロジック ---

function togglePosition() {
    const isBottom = container.classList.toggle('bottom-docked');
    storageSet(STORAGE_KEY_POSITION, isBottom);
    
    if (isBottom) {
        positionBtn.innerText = '▲';
        positionBtn.title = 'Move to Top';
    } else {
        positionBtn.innerText = '▼';
        positionBtn.title = 'Move to Bottom';
    }
    updateBodyMargin();
}

mainDateLabel.onmousedown = (e) => e.stopPropagation();
mainDateLabel.onclick = (e) => {
    e.stopPropagation();
    isCalendarView = true;
    render();
};


minimizeBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    toggleCollapse();
});

positionBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    togglePosition();
});

calendarBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    isCalendarView = !isCalendarView;
    if (isCalendarView) calendarOffsetPx = 0; // Reset to today when opening
    render();
});

// --- イベントリスナー ---
container.addEventListener('mousedown', (e) => {
    if (e.target.tagName === 'BUTTON') return;
    isDragging = true;
    dragStartX = e.clientX;
    dragStartCalendarOffsetPx = calendarOffsetPx;
    dragStartOffsetMs = manualOffsetMs;
    container.style.cursor = 'grabbing';
    if (revertTimer) { clearTimeout(revertTimer); revertTimer = null; }
});

container.addEventListener('click', () => {
    if (container.classList.contains('collapsed')) {
        toggleCollapse();
        return;
    }
    if (isCalendarView) {
        isCalendarView = false;
        render();
    }
});

window.addEventListener('mousemove', (e) => {
    if (!isDragging) return;
    const deltaX = e.clientX - dragStartX;
    
    if (isCalendarView) {
        calendarOffsetPx = dragStartCalendarOffsetPx + deltaX;
    } else {
        const deltaHours = -deltaX / PIXELS_PER_HOUR;
        manualOffsetMs = dragStartOffsetMs + (deltaHours * 3600 * 1000);
    }
});

window.addEventListener('mouseup', () => {
    if (!isDragging) return;
    isDragging = false;
    container.style.cursor = 'grab';
    if (!isCalendarView) {
        revertTimer = setTimeout(() => { manualOffsetMs = 0; }, 1000);
    }
});

// --- 設定UI ---
function openSettings() {
    overlay.style.display = 'block';
    modal.style.display = 'block';
    renderZoneList();
}
function closeSettings() {
    overlay.style.display = 'none';
    modal.style.display = 'none';
}
function renderZoneList() {
    zoneListEl.innerHTML = '';
    activeZones.forEach((z, index) => {
        const div = document.createElement('div');
        div.className = 'zone-item';
        div.innerHTML = `
            <span><span style="display:inline-block;width:10px;height:10px;background:${z.color || '#ff6b6b'};margin-right:5px;border-radius:50%;"></span>${z.label} <small style="color:#888">(${z.zone})</small></span>
            <button class="danger" data-index="${index}">×</button>
        `;
        zoneListEl.appendChild(div);
    });
}

document.getElementById('settings-btn').addEventListener('click', openSettings);
document.getElementById('close-settings-btn').addEventListener('click', closeSettings);
overlay.addEventListener('click', closeSettings);

zoneListEl.addEventListener('click', (e) => {
    if (e.target.tagName === 'BUTTON' && e.target.classList.contains('danger')) {
        activeZones.splice(e.target.dataset.index, 1);
        storageSet(STORAGE_KEY_ZONES, activeZones);
        renderZoneList();
    }
});

document.getElementById('add-zone-btn').addEventListener('click', () => {
    const label = document.getElementById('new-label').value.trim();
    const zone = document.getElementById('new-timezone').value;
    const color = document.getElementById('new-color').value;
    if (label && zone) {
        activeZones.push({ label, zone, color });
        document.getElementById('new-label').value = '';
        storageSet(STORAGE_KEY_ZONES, activeZones);
        renderZoneList();
    }
});

document.getElementById('bg-color-picker').addEventListener('input', (e) => {
    container.style.setProperty('--bg-color', e.target.value);
    storageSet(STORAGE_KEY_BG, e.target.value);
});

availableTimezones.forEach(tz => {
    const opt = document.createElement('option');
    opt.value = tz; opt.innerText = tz;
    tzSelect.appendChild(opt);
});

async function initFromStorage() {
    const isCollapsed = await storageGet(STORAGE_KEY_COLLAPSED, false);
    if (isCollapsed) container.classList.add('collapsed');

    const isBottom = await storageGet(STORAGE_KEY_POSITION, false);
    if (isBottom) {
        container.classList.add('bottom-docked');
        positionBtn.innerText = '▲';
        positionBtn.title = 'Move to Top';
    }

    updateBodyMargin();
    await loadSettings();
}

initFromStorage().finally(() => loop());
})();

